#!/bin/bash

#*****************************************************************************
#*****************************************************************************
# This script is not meant to be run standalone. It is always sourced by the
# master install script "sw"
#*****************************************************************************
#*****************************************************************************
(return 0 2>/dev/null) && sourced=1 || sourced=0


VMNG_CONFIG_LOC="/opt/web-app/etc"
VMNG_DST_LOC="/opt/data"
VMNG_EXTRA_PKGS="extra-packages.tar.gz"
VMNG_UPGRADE_CONTEXT="upgrade-context.json"
D_VMNG_EXTRA_PKGS="${VMNG_DST_LOC}/extra-packages"
VMNG_TOP_TEMP="${VMNG_DST_LOC}/temp/"
VMNG_TEMP="${VMNG_TOP_TEMP}/vmng/"
PYTHON2="/usr/bin/python2"
PYTHON3="/usr/bin/python3"

check_upgrade_context=1

sw_check_upgrade_context() {
    check_upgrade_context=1
    major_ver=$(cat /etc/version | cut -d "." -f 1)
    minor_ver=$(cat /etc/version | cut -d "." -f 2)
    if [ "$major_ver" -eq 20 ] && [ "$minor_ver" -ge 6 ]; then
        check_upgrade_context=0
    fi
    if [ "$major_ver" -gt 20 ]; then
        check_upgrade_context=0
    fi
    sw_log "${FUNCNAME[0]}: check_upgrade_context = $check_upgrade_context"
}

sw_hook_vmng_install() {
    local ver=$1

    # signature has been validated.
    # copy and extract extra-packages to the right path in /opt/data/
    local l_dest_dir="${D_VMNG_EXTRA_PKGS}/${ver}/"
    mkdir -p "$l_dest_dir"
    tar xzf $VMNG_TEMP/${VMNG_EXTRA_PKGS} -C "$l_dest_dir" --strip 1
    #store the original tar.gz for sw reset case
    mv $VMNG_TEMP/${VMNG_EXTRA_PKGS} $l_dest_dir

    sw_log "${FUNCNAME[0]}: Extracted contents of all ${VMNG_EXTRA_PKGS} to ${l_dest_dir}"

    rm -rf ${VMNG_TEMP}

    # Opportunistically call to prepare new version from the vantage point of the current version.
    # i.e. Don't die.
    sw_hook_prepare ${ver}

    return 0
}


# This as a sub-install script and the purpose of this script file is to
# complement/replace the master install script operations.
# During install or remove, if there are some behavioral changes to apply we do
# it in this sw_hook.sh script.

#
# These functions provide a hook into sw for install, remove operations
# The hook can do one of the following (& specific return code implies specific
# things as described below):
#
#   0. No-OP (return 0)
#     - For eg. Nothing specific to be done. The master install script "sw"
#       will continue with everything as is including untar for install and
#       remove all for remove case
#
# 101. Replace untar step in the operation so that personality specific things
#      can be done (return 101). In case of remove it could be additional removal from
#      additional partitions.
#     - For eg. in an install operation untar step alone is replaced and the steps
#       before and after are left as is. So "sw" will continue but assumes that
#       untar is done
#
# 102. Substitute the whole operation for a specific personality (return 102) from untar step
#     - For eg. takeover the complete install operation
#
#  -1. "Hell broke lose" so die...
#
sw_hook_extract()
{
    local d_temp_dir=$1 # destination dir where the img will be extracted
    local l_img=$2 # complete tar.gz
    local l_tar_tmp_file=$3 #contents of tar file

    local size=0
    local total_space=$(BLOCKSIZE=1 df "$d_temp_dir" | sed 's/^[^ ]* *\([0-9]*\) .*$/\1/')
    local avail_space=$(BLOCKSIZE=1 df "$d_temp_dir" | sed 's/^[^ ]* *[0-9]* *[0-9]* *\([0-9]*\) .*$/\1/')
    local one_pct=$((${total_space} / 100))
    local max_size=$((${avail_space} - ${one_pct}))

    for val in $(sed 's/^[^ ]* [^ ]* *\([0-9]*\) .*$/\1/' "${l_tar_tmp_file}"); do
        size=$(($size + $val))
    done

    sw_log "${FUNCNAME[0]}: Checking size ${size} against maxsize ${max_size}"
    if [ ${size} -gt ${max_size} ]; then
        sw_log "Error: Insufficient space in ${d_temp_dir} for ${l_img} to use. Make available ${size} bytes free before installation."
        die "Error: Disk space is not enough to install new image. Please set current software version as the default version and remove unused software versions first"
    fi

    if [ "x$sw_current_personality" == "xvmanage" ] && [[ ! -f /opt/web-app/etc/skip_downgrade_check ]]; then
        five_gb=$((5*1024*1024*1024))
        opt_data_sizeK=$(df --output=avail /opt/data/ | tail -1)
        opt_data_size=$((opt_data_sizeK * 1024))
        required_avail_space=$((${size} + ${five_gb}))
         if [ ${required_avail_space} -gt ${max_size} ]; then
              sw_log "required_avail_space is ${required_avail_space} and max_size is ${max_size}"
              sw_log "Error: Insufficient space in /opt/data for ${l_img} to use. Make available ${required_avail_space} bytes in /opt/data free before installation."
              die "Error: /opt/data space is not enough to install new image. Please set current software version as the default version and remove unused software versions first"
         fi
    fi

    mkdir "$d_temp_dir"/root
    omask=$(umask)
    umask 0077
    tar --no-same-permissions -oxzf "$l_img" -C "$d_temp_dir"/root
    l_ret_code=$?
    if [ $l_ret_code -eq 0 ]; then
        sw_log "${FUNCNAME[0]}: untar of image $l_img to $d_temp_dir successful"
        (cd "$d_temp_dir"/root && md5sum -c --quiet md5sum) || die "${FUNCNAME[0]}: Incorrect md5sum of image"
        umask "$omask"
    else
        umask "$omask"
        rm -rf "$d_temp_dir"/root
        die "Error: untar of image $l_img to $d_temp_dir failed"
    fi
}

sw_hook_vmng_move_and_validate()
{
    local l_tar_tmp_file="$1"

    local l_found_extra=$(grep ${VMNG_EXTRA_PKGS} "${l_tar_tmp_file}" | awk '{print $6}')

    if [ "x$l_found_extra" == "x${VMNG_EXTRA_PKGS}" ]; then
        sw_log "${FUNCNAME[0]}: Found the vmanage extra packages: $l_found_extra, extracting it"
        #Now do signature verification on the file
        validate $VMNG_TOP_TEMP/root "$l_found_extra"
        ret=$?
        if [ $ret -eq 0 ]; then
            sw_log "${FUNCNAME[0]}: Signature validation of ${VMNG_EXTRA} successful - ${VMNG_TEMP} ${l_found_extra}"
            mkdir -p $VMNG_TEMP
            mv $VMNG_TOP_TEMP/root/${VMNG_EXTRA_PKGS} $VMNG_TEMP
        else
            sw_log "${FUNCNAME[0]}: Error: signature validation of ${VMNG_EXTRA} failed - ${VMNG_TEMP} ${l_found_extra}"
            return 1
        fi
    fi
}

sw_hook_install()
{
    # arg1 - The file that has the tar ball contents listing
    # arg2 - The kinda operation install/verify/verify_vmanage
    # arg3 - The tar file extract destination directory
    # arg4 - The actuall tar ball image file
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific install stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw install case"
        local l_tar_tmp_file=$1
        local l_dest_dir=$3
        local l_img=$4
        rm -rf ${VMNG_TOP_TEMP}
        mkdir -p ${VMNG_TOP_TEMP}
        sw_hook_extract ${VMNG_TOP_TEMP} "$l_img" "$l_tar_tmp_file"

        #vmanage moves its contents and sig and cert files to vmanage specific temp file
        sw_hook_vmng_move_and_validate "$l_tar_tmp_file"
        ret=$?
        if [ $ret -eq 0 ]; then
            sw_log "$(basename "$0"): vmanage validation successful"
        else
            sw_log "$(basename "$0"): vmanage validation failed"
            rm -rf ${VMNG_TOP_TEMP}
            die "Error: signature validation of vmangage component failed"
        fi
        mount -o ro,loop ${VMNG_TOP_TEMP}/root/rootfs.img /mnt
        new_version=$(cat /mnt/etc/version)
        umount /mnt

        #Check if we have enough space in /boot to fit in the new image contents
        #max_size calculation is already done in "sw" so leverage that variable
        #to avoid confusion copying that to a local var
        local l_max_size=${max_size}
        local l_size=$(du -s --block-size=1 ${VMNG_TOP_TEMP}/root/ | awk '{print $1}')
        local l_num_files=$(cat "${l_tar_tmp_file}" | wc -l)
        sw_log "${FUNCNAME[0]}: Removing temp file ${l_tar_tmp_file} with ${l_num_files} total files"
        rm -f "${l_tar_tmp_file}"

        sw_log "${FUNCNAME[0]}: Checking size ${l_size} against maxsize ${l_max_size}"
        if [ ${l_size} -gt ${l_max_size} ]; then
            rm -rf ${VMNG_TOP_TEMP}
            sw_log "Error: Insufficient space in ${l_dest_dir} for ${l_img} to use. Make available ${l_size} bytes free before installation."
            die "Error: Disk space is not enough to install new image. Please set current software version as the default version and remove unused software versions first"
        fi
        sw_log "${FUNCNAME[0]}: Checking number of files ${l_num_files} against max of 50"
        if [ "${l_num_files}" -gt 50 ]; then
            rm -rf ${VMNG_TOP_TEMP}
            die "Error: tar contains ${l_num_files} files (exceeding the limit of 50)"
        fi

        #All clear. Go ahead with moving the files to dest_dir
        rm -rf $l_dest_dir/*
        mv ${VMNG_TOP_TEMP}/root/* $l_dest_dir/.

        sw_log "${FUNCNAME[0]}: Start installing version ${new_version}"
        sw_hook_vmng_install "${new_version}"

        rm -rf ${VMNG_TOP_TEMP}
        #Exit with with specific code to indicate whether "sw" script should bail out or not
        return 101
    fi
}

sw_hook_vmng_remove()
{
    #needs version to be removed
    local ver=$1

    sw_log "${FUNCNAME[0]}: removing vmanage extra-packages for ${ver}"
    rm -rf ${D_VMNG_EXTRA_PKGS}/"${ver}"

    return 0
}

sw_hook_vmng_activate() {
    local new_ver=$1
    local curr_ver=$(cat /etc/version)
    local upgrade_context="${D_VMNG_EXTRA_PKGS}/${new_ver}/${VMNG_UPGRADE_CONTEXT}"
    local dest_upgrade_context="${VMNG_CONFIG_LOC}/${VMNG_UPGRADE_CONTEXT}"
    local reset_file="/boot/reset"

    sw_log "activate: cur version $curr_ver next version $new_ver"

    # ============================================================================================
    # Begin activation-check and die early to terminate activation.
    # NOTE:
    #   THIS DOES NOT PREVENT ALL ACTIVATION PATHS. IN HYPOTHETICAL SCENARIO WHERE A DETERMINED
    #   USER CAN INSTALL, REBOOT, AND INTERRUPT BOOT TO SELECT THE NEWLY INSTALLED VERSION, ANY
    #   ACTIVATION-CHECK ENFORCED HERE WILL BE BY-PASSSED. THIS WOULD NOT BE A SUPPORTED USE-CASE
    #   OF VMANAGE UPGRADE - BUT NEVERTHELESS ACHIEVABLE BY A USER.
    # ============================================================================================

    sw_check_upgrade_context

    # During reset software, activate hook is called, if reset is present we skip context-check.
    if [ ! -f ${reset_file} ] && [ "$check_upgrade_context" -eq 1 ]; then
        # Check whether upgrade context is prepared.

        if [ ! -f ${upgrade_context} ]; then
            sw_log "${FUNCNAME[0]}: Missing upgrade context file (${upgrade_context})"
            sw_log "${FUNCNAME[0]}: Please re-prepare ${new_ver} for activation"

            # Note when executed in 20.5 or prior version, re-prepare => remove and reinstall the version.
            die "Missing upgrade context file ${upgrade_context}, please re-prepare ${new_ver} for activation"
        fi

        # Check if all required upgrade context data is present.
        local node_ids=$(jq '.entries | has("json:/cluster/node-id-map/v1")' ${upgrade_context})
        if [ "${node_ids}" != "true" ]; then
            sw_log "${FUNCNAME[0]}: Missing required data in context file (${upgrade_context})"
            sw_log "${FUNCNAME[0]}: Please re-prepare ${new_ver} for activation"

            # Note when executed in 20.5 or prior version, re-prepare => remove and reinstall the version.
            die "Missing required data in context file ${upgrade_context}, please re-prepare ${new_ver} for activation"
        fi
        # Copy the upgrade context from target version's package directory to designated location.
        cp "${upgrade_context}" "${dest_upgrade_context}" &> /dev/null
        chown vmanage:vmanage-admin "${dest_upgrade_context}" &> /dev/null
        chmod 664 "${dest_upgrade_context}" &> /dev/null
    fi

    # ============================================================================================
    # Begin activation in earnest.
    # NOTE: NO MORE DYING WITHOUT PERFORMING CLEANUP FIRST.
    # ============================================================================================
    sw_log "Stop config-db cleanly..."
    sv stop neo4j &> /dev/null
    sync
    sw_log "Stop wildfly..."
    sv stop wildfly &> /dev/null
    sync
    sleep 10

    [ -d ${D_VMNG_EXTRA_PKGS}/"${curr_ver}"/containers/ ] || return 0

    echo "1" > ${D_VMNG_EXTRA_PKGS}/"${new_ver}"/upgrade_required &> /dev/null
    sw_log "${FUNCNAME[0]}: Stopping and clearing container-manager for ${new_ver} activate"
    sv stop container-manager &> /dev/null
    sync
    rm -rf /var/lib/nms/docker/* &> /dev/null

    return 0
}

sw_hook_vmng_prepare()
{
    local target=$1
    local current=$(cat /etc/version)
    local target_dir="${D_VMNG_EXTRA_PKGS}/${target}"
    local upgrade_context="${target_dir}/${VMNG_UPGRADE_CONTEXT}"

    sw_log "prepare: current version $current target version $target"

    # Setup general permissions.
    chmod 755 "$D_VMNG_EXTRA_PKGS"
    chmod 755 "${target_dir}"
    chown --recursive root:root "${target_dir}"
    find "${target_dir}" -type d -exec chmod 755 {} \;
    find "${target_dir}" -type f -exec chmod 644 {} \;

    # Setup upgrade transfer context environment.
    # `admin` is the only default user, and de-facto owner of the context file.
    # `vmanage-admin` is the default service-user's group, and group owner of the context file.
    # context file is read/write for owner user and group, and world-readable.
    echo "{}" > "${upgrade_context}"
    chmod 664 "${upgrade_context}"
    chown admin:vmanage-admin "${upgrade_context}"

    # Clean up current version extra_pkgs.tar.gz to free up /boot space
    local extra_file_list_str=$(find /boot -name ${VMNG_EXTRA_PKGS})
    local extra_file_list=(${extra_file_list_str})
    for extra_file in "${extra_file_list[@]}"; do
        extra_file_path=$(dirname "${extra_file}")
        extra_file_version=$(basename "${extra_file_path}")
        mkdir -p "${D_VMNG_EXTRA_PKGS}/${extra_file_version}/"
        mv -f "${extra_file}" "${D_VMNG_EXTRA_PKGS}/${extra_file_version}/"
        sw_log "prepare: moved ${extra_file} to ${D_VMNG_EXTRA_PKGS}/${extra_file_version}/"
    done

    #######################################################################################################
    # A little of background for this portion of the code:
    # 20.6 cluster has been totally restructured and in order to have enough information to bootstrap the
    # 20.6 from prior releases, some info from prior releases need to be available in all nodes before the
    # respective activation happens to avoid corner cases (please refer to sdwan/poky#2424 for details).
    # The only way to gate this is to run the script to gather info during install.
    # Options to execute pre-check prior to 20.6 were explored but nothing was available. It seems that from  
    # one node, there is no easy access to the other nodes during any part of sw execution that does not 
    # require access credential from user to login to the remote nodes.
    # There is a unexplored option that requires further investigation and is dependent on vManage able 
    # to establish SSH to remote peers using existing key-files for auth.
    # And in order to avoid an extra step by the user, the upgrade script is being called here. There is
    # no guarantee that will work but it is a best effort to make the upgrade easier.
    # Because different versions have different python versions, and python3 could be partially 
    # installed and not really functional, we go for the best effort, if running python2 succeeds that is the
    # right version on the box, otherwise we try python3.

    # if python2 does not exist or produce error, try python3
    sw_check_upgrade_context
    if [ "$check_upgrade_context" -eq 1 ]; then
        var=$(python -c 'import json' 2>&1 >/dev/null)
        [[ $? -eq 0 ]] && ${PYTHON2} "${target_dir}/extra/bin/upgrade-context-py2.py" || ${PYTHON3} "${target_dir}/extra/bin/upgrade-context.py"
        return $?
    fi
    return 0
}

sw_hook_remove()
{
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific cleanup stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw remove"
        local ver=$1
        sw_hook_vmng_remove "$ver"
        #Please read the comments above on what return code should be returned
        return 0
    fi
}

sw_hook_verify()
{
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific cleanup stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw remove case Nothing to be done"
        #Please read the comments above on what return code should be returned
        return 0
    fi
}

sw_hook_verify_vmanage()
{
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific cleanup stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw remove case Nothing to be done"
        #Please read the comments above on what return code should be returned
        return 0
    fi
}

#This function removes viptelatac user from the system
sw_hook_remove_viptelatacuser()
{
    if [ -z "$(grep viptelatac /etc/passwd)" ]; then
        return
    fi
    #Remove viptelatac user
    sw_log "${FUNCNAME[0]}: Deleting viptelatac user"
    #Remove from cdb as well
    local RETRIES=0
    while : ; do
        local ret_code=0
        /usr/bin/confd_cmd -c "mctrans;mdel /system/aaa/user{viptelatac};mcommit;ccommit"
        ret_code=$?
        if [ ${ret_code} -eq 0 ]; then
            sw_log "${FUNCNAME[0]}: viptelatac user successfully removed from cdb"
            break
        else
            sw_log "${FUNCNAME[0]}: Retry deleting viptelatac user from cdb. Err: ${ret_code}"
            RETRIES=$((RETRIES + 1))
            if [ ${RETRIES} -gt 10 ]; then
                sw_log "${FUNCNAME[0]}: Error removing viptelatac user (max cdb retries done). Remove manually"
                break
            else
                sleep 0.5
            fi
        fi
    done
}

#This function performs any task that is common across all personalities
sw_hook_activate_common()
{
    sw_hook_remove_viptelatacuser
}

sw_hook_activate()
{
    #Invoke the common stuff across all personalities
    sw_hook_activate_common
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        sw_log "$(basename "$0"): $@ - vmanage sw activate"
        sw_hook_vmng_activate "$1"
        #Please read the comments above on what return code should be returned
        return $?
    fi
}

sw_hook_prepare()
{
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        sw_log "$(basename "$0"): $@ - vmanage sw prepare case"
        sw_hook_vmng_prepare "$1"
        return $?
    fi
}
