#!/bin/bash

#*****************************************************************************
#*****************************************************************************
# This script is not meant to be run standalone. It is always sourced by the
# master install script "sw"
#*****************************************************************************
#*****************************************************************************
(return 0 2>/dev/null) && sourced=1 || sourced=0


VMNG_DST_LOC="/opt/data"
VMNG_EXTRA_PKGS="extra-packages.tar.gz"
D_VMNG_EXTRA_PKGS="${VMNG_DST_LOC}/extra-packages"
VMNG_TOP_TEMP="${VMNG_DST_LOC}/temp/"
VMNG_TEMP="${VMNG_TOP_TEMP}/vmng/"

sw_hook_vmng_install() {
    local ver=$1

    # signature has been validated.
    # copy and extract extra-packages to the right path in /opt/data/
    local l_dest_dir="${D_VMNG_EXTRA_PKGS}/${ver}/"
    mkdir -p "$l_dest_dir"
    tar xzf $VMNG_TEMP/${VMNG_EXTRA_PKGS} -C "$l_dest_dir" --strip 1
    sw_log "${FUNCNAME[0]}: extracted ${VMNG_EXTRA_PKGS} to ${l_dest_dir}"
    for i in $(find "${l_dest_dir}" -name '*.tar.gz')
    do
        sw_log "${FUNCNAME[0]}: extracting $i"
        l_dirname="$(dirname "${i}")"
        tar xzf "$i" -C "${l_dirname}"
        rm "$i"
    done
    #store the original tar.gz for sw reset case
    mv $VMNG_TEMP/${VMNG_EXTRA_PKGS} $l_dest_dir

    sw_log "${FUNCNAME[0]}: Extracted contents of all ${VMNG_EXTRA_PKGS} to ${l_dest_dir}"

    rm -rf ${VMNG_TEMP}
}


# This as a sub-install script and the purpose of this script file is to
# complement/replace the master install script operations.
# During install or remove, if there are some behavioral changes to apply we do
# it in this sw_hook.sh script.

#
# These functions provide a hook into sw for install, remove operations
# The hook can do one of the following (& specific return code implies specific
# things as described below):
#
#   0. No-OP (return 0)
#     - For eg. Nothing specific to be done. The master install script "sw"
#       will continue with everything as is including untar for install and
#       remove all for remove case
#
# 101. Replace untar step in the operation so that personality specific things
#      can be done (return 101). In case of remove it could be additional removal from
#      additional partitions.
#     - For eg. in an install operation untar step alone is replaced and the steps
#       before and after are left as is. So "sw" will continue but assumes that
#       untar is done
#
# 102. Substitute the whole operation for a specific personality (return 102) from untar step
#     - For eg. takeover the complete install operation
#
#  -1. "Hell broke lose" so die...
#
sw_hook_extract()
{
    local d_temp_dir=$1 # destination dir where the img will be extracted
    local l_img=$2 # complete tar.gz
    local l_tar_tmp_file=$3 #contents of tar file

    local size=0
    local total_space=$(BLOCKSIZE=1 df "$d_temp_dir" | sed 's/^[^ ]* *\([0-9]*\) .*$/\1/')
    local avail_space=$(BLOCKSIZE=1 df "$d_temp_dir" | sed 's/^[^ ]* *[0-9]* *[0-9]* *\([0-9]*\) .*$/\1/')
    local one_pct=$((${total_space} / 100))
    local max_size=$((${avail_space} - ${one_pct}))

    for val in $(sed 's/^[^ ]* [^ ]* *\([0-9]*\) .*$/\1/' "${l_tar_tmp_file}"); do
        size=$(($size + $val))
    done

    sw_log "${FUNCNAME[0]}: Checking size ${size} against maxsize ${max_size}"
    if [ ${size} -gt ${max_size} ]; then
        die "Error: tar contents are too large."
    fi

    mkdir "$d_temp_dir"/root
    omask=$(umask)
    umask 0077
    tar --no-same-permissions -oxzf "$l_img" -C "$d_temp_dir"/root
    l_ret_code=$?
    if [ $l_ret_code -eq 0 ]; then
        sw_log "${FUNCNAME[0]}: untar of image $l_img to $d_temp_dir successful"
        (cd "$d_temp_dir"/root && md5sum -c --quiet md5sum) || die "${FUNCNAME[0]}: Incorrect md5sum"
        umask "$omask"
    else
        umask "$omask"
        rm -rf "$d_temp_dir"/root
        die "Error: untar of image $l_img to $d_temp_dir failed"
    fi
}

sw_hook_vmng_move_and_validate()
{
    local l_tar_tmp_file="$1"

    local l_found_extra=$(grep ${VMNG_EXTRA_PKGS} "${l_tar_tmp_file}" | awk '{print $6}')

    if [ "x$l_found_extra" == "x${VMNG_EXTRA_PKGS}" ]; then
        sw_log "${FUNCNAME[0]}: Found the vmanage extra packages: $l_found_extra, extracting it"
        #Now do signature verification on the file
        validate $VMNG_TOP_TEMP/root "$l_found_extra"
        ret=$?
        if [ $ret -eq 0 ]; then
            sw_log "${FUNCNAME[0]}: Signature validation of ${VMNG_EXTRA} successful - ${VMNG_TEMP} ${l_found_extra}"
            mkdir -p $VMNG_TEMP
            mv $VMNG_TOP_TEMP/root/${VMNG_EXTRA_PKGS} $VMNG_TEMP
        else
            sw_log "${FUNCNAME[0]}: Error: signature validation of ${VMNG_EXTRA} failed - ${VMNG_TEMP} ${l_found_extra}"
            return 1
        fi
    fi
}

sw_hook_install()
{
    # arg1 - The file that has the tar ball contents listing
    # arg2 - The kinda operation install/verify/verify_vmanage
    # arg3 - The tar file extract destination directory
    # arg4 - The actuall tar ball image file
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific install stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw install case"
        local l_tar_tmp_file=$1
        local l_dest_dir=$3
        local l_img=$4
        rm -rf ${VMNG_TOP_TEMP}
        mkdir -p ${VMNG_TOP_TEMP}
        sw_hook_extract ${VMNG_TOP_TEMP} "$l_img" "$l_tar_tmp_file"

        #vmanage moves its contents and sig and cert files to vmanage specific temp file
        sw_hook_vmng_move_and_validate "$l_tar_tmp_file"
        ret=$?
        if [ $ret -eq 0 ]; then
            sw_log "$(basename "$0"): vmanage validation successful"
        else
            sw_log "$(basename "$0"): vmanage validation failed"
            rm -rf ${VMNG_TOP_TEMP}
            die "Error: signature validation of vmangage component failed"
        fi
        mount -o ro,loop ${VMNG_TOP_TEMP}/root/rootfs.img /mnt
        ver=$(cat /mnt/etc/version)
        umount /mnt

        #Check if we have enough space in /boot to fit in the new image contents
        #max_size calculation is already done in "sw" so leverage that variable
        #to avoid confusion copying that to a local var
        local l_max_size=${max_size}
        local l_size=$(du -s --block-size=1 ${VMNG_TOP_TEMP}/root/ | awk '{print $1}')
        local l_num_files=$(cat "${l_tar_tmp_file}" | wc -l)
        sw_log "${FUNCNAME[0]}: Removing temp file ${l_tar_tmp_file} with ${l_num_files} total files"
        rm -f "${l_tar_tmp_file}"

        sw_log "${FUNCNAME[0]}: Checking size ${l_size} against maxsize ${l_max_size}"
        if [ ${l_size} -gt ${l_max_size} ]; then
            rm -rf ${VMNG_TOP_TEMP}
            die "Error: tar contents are too large."
        fi
        sw_log "${FUNCNAME[0]}: Checking number of files ${l_num_files} against max of 50"
        if [ "${l_num_files}" -gt 50 ]; then
            rm -rf ${VMNG_TOP_TEMP}
            die "Error: tar contains too many files."
        fi

        #All clear. Go ahead with moving the files to dest_dir
        rm -rf $l_dest_dir/*
        mv ${VMNG_TOP_TEMP}/root/* $l_dest_dir/.

        sw_hook_vmng_install "$ver"

        rm -rf ${VMNG_TOP_TEMP}
        #Exit with with specific code to indicate whether "sw" script should bail out or not
        return 101
    fi
}

sw_hook_vmng_remove()
{
    #needs version to be removed
    local ver=$1

    sw_log "${FUNCNAME[0]}: removing vmanage extra-packages for ${ver}"
    rm -rf ${D_VMNG_EXTRA_PKGS}/"${ver}"

    return 0
}

sw_hook_vmng_activate() {
    # needs new version
    # needs old version
    local new_ver=$1

    local curr_ver=$(cat /etc/version)

    sw_log "${FUNCNAME[0]}: activating vmanage extra-packages for ${new_ver}"

    # first stop all the dockers that may be running in current version
    [ -d ${D_VMNG_EXTRA_PKGS}/"${curr_ver}"/containers/ ] || return 0
    for i in $(ls ${D_VMNG_EXTRA_PKGS}/"${curr_ver}"/containers/)
    do
        sw_log "${FUNCNAME[0]}: deactivating $i"
        /usr/bin/vconfd_container_management.py deactivate "$i" &> /dev/null
        # copy over the current run.meta file which holds the enable or ip information to new version
        [ -f ${D_VMNG_EXTRA_PKGS}/"${curr_ver}"/containers/"$i"/run.meta ] &&
          [ -d ${D_VMNG_EXTRA_PKGS}/"${new_ver}"/containers/"$i"/ ] &&
          cp ${D_VMNG_EXTRA_PKGS}/"${curr_ver}"/containers/"$i"/run.meta ${D_VMNG_EXTRA_PKGS}/"${new_ver}"/containers/"$i"/run.meta &> /dev/null
    done
    echo "1" > ${D_VMNG_EXTRA_PKGS}/"${new_ver}"/upgrade_required &> /dev/null
    sw_log "${FUNCNAME[0]}: stopping and clearing container-manager for ${new_ver} activate"
    sv stop container-manager &> /dev/null
    sync
    rm -rf /var/lib/nms/docker/* &> /dev/null

    return 0
}

sw_hook_remove()
{
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific cleanup stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw remove"
        local ver=$1
        sw_hook_vmng_remove "$ver"
        #Please read the comments above on what return code should be returned
        return 0
    fi
}

sw_hook_verify()
{
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific cleanup stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw remove case Nothing to be done"
        #Please read the comments above on what return code should be returned
        return 0
    fi
}

sw_hook_verify_vmanage()
{
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific cleanup stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw remove case Nothing to be done"
        #Please read the comments above on what return code should be returned
        return 0
    fi
}

#This function removes viptelatac user from the system
sw_hook_remove_viptelatacuser()
{
    if [ -z "$(grep viptelatac /etc/passwd)" ]; then
        return
    fi
    #Remove viptelatac user
    sw_log "${FUNCNAME[0]}: Deleting viptelatac user"
    #Remove from cdb as well
    local RETRIES=0
    while : ; do
        local ret_code=0
        /usr/bin/confd_cmd -c "mctrans;mdel /system/aaa/user{viptelatac};mcommit;ccommit"
        ret_code=$?
        if [ ${ret_code} -eq 0 ]; then
            sw_log "${FUNCNAME[0]}: viptelatac user successfully removed from cdb"
            break
        else
            sw_log "${FUNCNAME[0]}: Retry deleting viptelatac user from cdb. Err: ${ret_code}"
            RETRIES=$((RETRIES + 1))
            if [ ${RETRIES} -gt 10 ]; then
                sw_log "${FUNCNAME[0]}: Error removing viptelatac user (max cdb retries done). Remove manually"
                break
            else
                sleep 0.5
            fi
        fi
    done
}

#This function performs any task that is common across all personalities
sw_hook_activate_common()
{
    sw_hook_remove_viptelatacuser
}

sw_hook_activate()
{
    #Invoke the common stuff across all personalities
    sw_hook_activate_common
    if [ "x$sw_current_personality" == "xvmanage" ]; then
        # vmanage specific cleanup stuff goes in here
        sw_log "$(basename "$0"): $@ - vmanage sw activate"
        sw_hook_vmng_activate "$1"
        #Please read the comments above on what return code should be returned
        return 0
    fi
}
